/****************************************************************************
*
*    Copyright (c) 2005 - 2014 by Vivante Corp.  All rights reserved.
*
*    The material in this file is confidential and contains trade secrets
*    of Vivante Corporation. This is proprietary information owned by
*    Vivante Corporation. No part of this work may be disclosed,
*    reproduced, copied, transmitted, or used in any way for any purpose,
*    without the express written permission of Vivante Corporation.
*
*****************************************************************************/


#include "tiny_ui_platform.h"
#include "tiny_ui_kernel.h"
#include "tiny_ui_hal.h"
#include "tiny_ui_hw.h"

// profiling DB.
//#include "..\src\FSLtime.h"
//#include "mc_parameters.h"

//extern uint32_t Timing[16];  // profiling code DB.

static int s_reference = 0;
static tiny_ui_error_t do_terminate(tiny_ui_kernel_terminate_t * data);

struct tiny_ui_kernel_context {
    // Command buffer.
    void * command_buffer;
    void * command_buffer_logical;
    uint32_t command_buffer_physical;

    // Tessellation buffer.
    void * tessellation_buffer;
    void * tessellation_buffer_logical;
    uint32_t tessellation_buffer_physical;
};

static void gpu(int enable)
{
    tiny_ui_hw_clock_control_t value;

    if (enable) {
        // Disable clock gating.
        *(uint32_t *) &value = tiny_ui_hal_peek(TINY_UI_HW_CLOCK_CONTROL);
        value.clock_gate = 0;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(1);

        // Set clock speed.
        value.scale = 64;
        value.scale_load = 1;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(1);
        value.scale_load = 0;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(5);

        do {
            // Perform a soft reset.
            value.isolate = 1;
            tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
            value.soft_reset = 1;
            tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
            tiny_ui_hal_delay(5);
            value.soft_reset = 0;
            tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
            value.isolate = 0;
            tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        } while ((tiny_ui_hal_peek(TINY_UI_HW_IDLE) & TINY_UI_HW_IDLE_STATE) != TINY_UI_HW_IDLE_STATE);
    } else {
        // Wait for idle.
        while ((tiny_ui_hal_peek(TINY_UI_HW_IDLE) & TINY_UI_HW_IDLE_STATE) != TINY_UI_HW_IDLE_STATE) ;

        // Set idle speed.
        *(uint32_t *) &value = tiny_ui_hal_peek(TINY_UI_HW_CLOCK_CONTROL);
        value.scale = 1;
        value.scale_load = 1;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(1);
        value.scale_load = 0;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(5);

        // Enable clock gating.
        value.clock_gate = 1;
        tiny_ui_hal_poke(TINY_UI_HW_CLOCK_CONTROL, *(uint32_t *) &value);
        tiny_ui_hal_delay(1);
    }
}

static tiny_ui_error_t do_initialize(tiny_ui_kernel_initialize_t * data)
{
    tiny_ui_kernel_context_t * context;
    uint32_t id;
    uint32_t extraCaps;

    // Construct the context.
    context = (tiny_ui_kernel_context_t *) tiny_ui_hal_alloc(sizeof(tiny_ui_kernel_context_t));
    if (context == NULL) {
        // Out of memory.
        return TINY_UI_OUT_OF_MEMORY;
    }

    // Zero out all pointers.
    context->command_buffer = NULL;
    context->tessellation_buffer = NULL;

    // Increment reference counter.
    if (s_reference++ == 0) {
        // Initialize the SOC.
        tiny_ui_hal_initialize();

        // Enable the GPU.
        gpu(1);
    }

    // Fill in hardware capabilities.
    *(uint32_t *) &data->capabilities = 0;
    id = tiny_ui_hal_peek(TINY_UI_HW_CHIP_ID);
    if (id >= 0x300) {
        data->capabilities.l2_cache = 1;
    }
    extraCaps = tiny_ui_hal_peek(TINY_UI_HW_FEATURES) & TINY_UI_HW_FEATURE_UI;
    if (extraCaps) {
        data->capabilities.l8 = 1;
        data->capabilities.border_culling = 1;
    }

    // Allocate the command buffer.
    if (data->command_buffer_size) {
        // Allocate the memory.
        context->command_buffer = tiny_ui_hal_allocate_contiguous(data->command_buffer_size,
                                                                  &context->command_buffer_logical,
                                                                  &context->command_buffer_physical);
        if (context->command_buffer == NULL) {
            // Free any allocated memory.
            tiny_ui_kernel_terminate_t terminate = { context };
            do_terminate(&terminate);

            // Out of memory.
            return TINY_UI_OUT_OF_MEMORY;
        }

        // Return command buffer logical pointer and GPU address.
        data->command_buffer = context->command_buffer_logical;
        data->command_buffer_gpu = context->command_buffer_physical;
    }

    // Allocate the tessellation buffer.
    if ((data->tessellation_width > 0) && (data->tessellation_height > 0)) {
        // Allign tesselation size to 16.
        int width = (data->tessellation_width + 15) & ~15;
        int height = (data->tessellation_height + 15) & ~15;
        unsigned long stride, buffer_size, l1_size, l2_size;

        if (!extraCaps) {
            // Align to 128x16.
            width = (width + 127) & ~127;
        }

        // Check if we can used tiled tessellation (128x16).
        if (((width & 127) == 0) && ((height & 15) == 0)) {
            data->capabilities.tiled = 0x3;
        } else {
            data->capabilities.tiled = 0x2;
        }

        // Compute tessellation buffer size.
        stride = TINY_UI_ALIGN(width * 8, 64);
        buffer_size = TINY_UI_ALIGN(stride * height, 64);
        // Each bit in the L1 cache represents 64 bytes of tessellation data.
        l1_size = TINY_UI_ALIGN(TINY_UI_ALIGN(buffer_size / 64, 64) / 8, 64);
        // Each bit in the L2 cache represents 32 bytes of L1 data.
        l2_size = data->capabilities.l2_cache ? TINY_UI_ALIGN(TINY_UI_ALIGN(l1_size / 32, 64) / 8, 64) : 0;

        // Allocate the memory.
        context->tessellation_buffer = tiny_ui_hal_allocate_contiguous(buffer_size + l1_size + l2_size,
                                                                       &context->tessellation_buffer_logical,
                                                                       &context->tessellation_buffer_physical);
        if (context->tessellation_buffer == NULL) {
            // Free any allocated memory.
            tiny_ui_kernel_terminate_t terminate = { context };
            do_terminate(&terminate);

            // Out of memory.
            return TINY_UI_OUT_OF_MEMORY;
        }

        // Return the tessellation buffer pointers and GPU addresses.
        data->tessellation_buffer_gpu[0] = context->tessellation_buffer_physical;
        data->tessellation_buffer_gpu[1] = context->tessellation_buffer_physical + buffer_size;
        data->tessellation_buffer_gpu[2] = (l2_size ? data->tessellation_buffer_gpu[1] + l1_size
                                            : data->tessellation_buffer_gpu[1]);
        data->tessellation_buffer_size[0] = buffer_size;
        data->tessellation_buffer_size[1] = l1_size;
        data->tessellation_buffer_size[2] = l2_size;
        data->tessellation_stride = stride;
        data->tessellation_width_height = width | (height << 16);
    }

    // Return context pointer.
    data->context = context;

    // Enable all interrupts.
    tiny_ui_hal_poke(TINY_UI_INTR_ENABLE, 0xFFFFFFFF);

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_terminate(tiny_ui_kernel_terminate_t * data)
{
    // Free any allocated memory for the context.
    if (data->context->command_buffer) {
        // Free the command buffer.
        tiny_ui_hal_free_contiguous(data->context->command_buffer);
    }

    if (data->context->tessellation_buffer) {
        // Free the tessellation buffer.
        tiny_ui_hal_free_contiguous(data->context->tessellation_buffer);
    }

    // Free the context.
    tiny_ui_hal_free(data->context);

    // Decrement reference counter.
    if (--s_reference == 0) {
        // Disable the GPU.
        gpu(0);

        // De-initialize the SOC.
        tiny_ui_hal_deinitialize();
    }

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_allocate(tiny_ui_kernel_allocate_t * data)
{
    // Allocate the memory.
    data->memory_handle = tiny_ui_hal_allocate_contiguous(data->bytes, &data->memory, &data->memory_gpu);
    if (data->memory_handle == NULL) {
        // Out of memory.
        return TINY_UI_OUT_OF_MEMORY;
    }

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_free(tiny_ui_kernel_free_t * data)
{
    // Free the memory.
    tiny_ui_hal_free_contiguous(data->memory_handle);

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_submit(tiny_ui_kernel_submit_t * data)
{
    uint32_t offset;

    // Perform a memory barrier.
    tiny_ui_hal_barrier();

    offset = (uint8_t *) data->commands - (uint8_t *) data->context->command_buffer_logical;

    // Write the registers.
    tiny_ui_hal_poke(TINY_UI_HW_CMDBUF_ADDRESS, data->context->command_buffer_physical + offset);
    tiny_ui_hal_poke(TINY_UI_HW_CMDBUF_SIZE, (data->command_size + 7) / 8);

    //profiling code DB. 
        //Timing[14]=FSLtime.ms;
        //Timing[15]= var32_read(PIT_CVAL1) / 66;
    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_wait(tiny_ui_kernel_wait_t * data)
{
    // Wait for C-Model interrupt.
    if (!tiny_ui_hal_wait_interrupt(data->timeout_ms)) {
        // Timeout.
        return TINY_UI_TIMEOUT;
    }

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_reset(tiny_ui_kernel_reset_t * data)
{
    // Disable and enable the GPU.
    gpu(0);
    gpu(1);

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_debug(tiny_ui_kernel_debug_t * data)
{
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_map(tiny_ui_kernel_map_t * data)
{
    // Map the memory.
    data->memory_handle = tiny_ui_hal_map(data->bytes, data->logical, data->physical, &data->memory_gpu);
    if (data->memory_handle == NULL)
    {
        // Out of resources.
        return TINY_UI_OUT_OF_RESOURCES;
    }

    // Success.
    return TINY_UI_SUCCESS;
}

static tiny_ui_error_t do_unmap(tiny_ui_kernel_unmap_t * data)
{
    // Unmap the memory.
    tiny_ui_hal_unmap(data->memory_handle);

    // Success.
    return TINY_UI_SUCCESS;
}

tiny_ui_error_t tiny_ui_kernel(tiny_ui_kernel_command_t command, void * data)
{
    // Dispatch on command.
    switch (command) {
        case TINY_UI_INITIALIZE:
            // Initialize the context.
            return do_initialize(data);

        case TINY_UI_TERMINATE:
            // Terminate the context.
            return do_terminate(data);

        case TINY_UI_ALLOCATE:
            // Allocate contiguous memory.
            return do_allocate(data);

        case TINY_UI_FREE:
            // Free contiguous memory.
            return do_free(data);

        case TINY_UI_SUBMIT:
            // Submit a command buffer.
            return do_submit(data);

        case TINY_UI_WAIT:
            // Wait for the GPU.
            return do_wait(data);

        case TINY_UI_RESET:
            // Reste the GPU.
            return do_reset(data);

        case TINY_UI_DEBUG:
            // Perform debugging features.
            return do_debug(data);

        case TINY_UI_MAP:
            // Map some memory.
            return do_map(data);

        case TINY_UI_UNMAP:
            // Unmap some memory.
            return do_unmap(data);
    }

    // Invalid command.
    return TINY_UI_INVALID_ARGUMENT;
}
